   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*             CLIPS Version 6.30  10/19/06            */
   /*                                                     */
   /*                    DRIVE MODULE                     */
   /*******************************************************/

/*************************************************************/
/* Purpose: Handles join network activity associated with    */
/*   with the addition of a data entity such as a fact or    */
/*   instance.                                               */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*      6.23: Correction for FalseSymbol/TrueSymbol. DR0859  */
/*                                                           */
/*      6.24: Removed INCREMENTAL_RESET and                  */
/*            LOGICAL_DEPENDENCIES compilation flags.        */
/*                                                           */
/*            Renamed BOOLEAN macro type to intBool.         */
/*                                                           */
/*            Rule with exists CE has incorrect activation.  */
/*            DR0867                                         */
/*                                                           */
/*      6.30: Added support for hashed alpha memories.       */
/*                                                           */
/*            Added additional developer statistics to help  */
/*            analyze join network performance.              */
/*                                                           */
/*            Removed pseudo-facts used in not CE.           */
/*                                                           */
/*************************************************************/

#define _DRIVE_SOURCE_

#include <stdio.h>
#define _STDIO_INCLUDED_
#include <stdlib.h>

#include "setup.h"

#if DEFRULE_CONSTRUCT

#include "agenda.h"
#include "constant.h"
#include "engine.h"
#include "envrnmnt.h"
#include "memalloc.h"
#include "prntutil.h"
#include "reteutil.h"
#include "retract.h"
#include "router.h"
#include "lgcldpnd.h"
#include "incrrset.h"

#include "drive.h"

struct compareFrame /* TBD Keep or Toss? */
  {
   DATA_OBJECT slotValue;
  };
  
  
/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

   static void                    PPDrive(void *,struct partialMatch *,struct partialMatch *,struct joinNode *);
   static void                    PNRDrive(void *,struct joinNode *,struct partialMatch *,
                                           struct partialMatch *);
   static void                    EmptyDrive(void *,struct joinNode *,struct partialMatch *);
   static void                    JoinNetErrorMessage(void *,struct joinNode *);
   static unsigned long           BetaMemoryHashValue(void *,struct expr *,struct partialMatch *,struct partialMatch *,struct joinNode *,struct compareFrame *);
   static struct compareFrame    *RetrieveStaticBinds(void *,struct partialMatch *,int,struct joinNode *,unsigned long *,int *);
   static struct compareFrame    *CreateDynamicBinds(void *theEnv,int);
   static int                     CompareFrames(void *,struct joinNode *,struct compareFrame *,struct compareFrame *,
                                                struct partialMatch *,struct partialMatch *,struct partialMatch *,struct expr *,int);
   static void                    NetworkAssertNegatedRight(void *,struct partialMatch *,struct joinNode *);
  
/************************************************/
/* NetworkAssert: Primary routine for filtering */
/*   a partial match through the join network.  */
/************************************************/
globle void NetworkAssert(
  void *theEnv,
  struct partialMatch *binds,
  struct joinNode *join)
  {
   /*=========================================================*/
   /* If an incremental reset is being performed and the join */
   /* is not part of the network to be reset, then return.    */
   /*=========================================================*/

#if (! BLOAD_ONLY) && (! RUN_TIME)
   if (EngineData(theEnv)->IncrementalResetInProgress && (join->initialize == FALSE)) return;
#endif

   /*==================================================*/
   /* Use a special routine if this is the first join. */
   /*==================================================*/

   if (join->firstJoin)
     {
      EmptyDrive(theEnv,join,binds);
      return;
     }

   /*================================*/
   /* Enter the join from the right. */
   /*================================*/

   if (join->patternIsNegated)
     { NetworkAssertNegatedRight(theEnv,binds,join); }
   else
     { NetworkAssertRight(theEnv,binds,join); }

   return;
  }

/*****************************************************/
/* NetworkAssertRight: Primary routine for filtering */
/*   a partial match through the join network.       */
/*****************************************************/
globle void NetworkAssertRight(
  void *theEnv,
  struct partialMatch *rhsBinds,
  struct joinNode *join)
  {
   struct partialMatch *lhsBinds = NULL;
   int exprResult;
   /*
   int frameSize = 0;
   struct compareFrame *staticFrame, *dynamicFrame;
   int needToEval;
   struct expr *leftHash, *rightHash;
*/
   /*=========================================================*/
   /* If an incremental reset is being performed and the join */
   /* is not part of the network to be reset, then return.    */
   /*=========================================================*/

#if (! BLOAD_ONLY) && (! RUN_TIME)
   if (EngineData(theEnv)->IncrementalResetInProgress && (join->initialize == FALSE)) return;
#endif

   /*==================================================*/
   /* Initialize some variables used to indicate which */
   /* side is being compared to the new partial match. */
   /*==================================================*/

   /*=====================================================*/
   /* The partial matches entering from the LHS of a join */
   /* are stored in the beta memory of the previous join  */
   /* (unless the current join is a join from the right   */
   /* or is attached to a not CE).                        */
   /*=====================================================*/

   lhsBinds = join->lastLevel->beta;
         
#if DEVELOPER
   if (lhsBinds != NULL)
     { EngineData(theEnv)->rightToLeftLoops++; }
#endif
    
   /*===================================================*/
   /* Compare each set of binds on the opposite side of */
   /* the join with the set of binds that entered this  */
   /* join. If the binds don't mismatch, then perform   */
   /* the appropriate action for the logic of the join. */
   /*===================================================*/

   while (lhsBinds != NULL)
     {      
      /*===========================================================*/
      /* Initialize some variables pointing to the partial matches */
      /* in the LHS and RHS of the join.                           */
      /*===========================================================*/

      if ((lhsBinds->hashExpr == join->leftHash) &&
          (lhsBinds->hashValue != rhsBinds->hashValue))
        {
         lhsBinds = lhsBinds->nextInNode;
#if DEVELOPER
         EngineData(theEnv)->betaHashSkips++;
#endif
         continue;
        }

      /*===================================================*/
      /* If the join has no expression associated with it, */
      /* then the new partial match derived from the LHS   */
      /* and RHS partial matches is valid.                 */
      /*===================================================*/

      if (join->networkTest == NULL)
        { exprResult = TRUE; }

      /*=========================================================*/
      /* If the join has an expression associated with it, then  */
      /* evaluate the expression to determine if the new partial */
      /* match derived from the LHS and RHS partial matches is   */
      /* valid (i.e. variable bindings are consistent and        */
      /* predicate expressions evaluate to TRUE).                */
      /*=========================================================*/

      else
        {
#if DEVELOPER
         EngineData(theEnv)->rightToLeftComparisons++;
#endif
         exprResult = EvaluateJoinExpression(theEnv,join->networkTest,lhsBinds,rhsBinds,join);
         if (EvaluationData(theEnv)->EvaluationError)
           {
            if (join->patternIsNegated) exprResult = TRUE;
            SetEvaluationError(theEnv,FALSE);
           }

#if DEVELOPER
         if (exprResult)
           { EngineData(theEnv)->rightToLeftSucceeds++; }
#endif
        }

      /*====================================================*/
      /* If the join expression evaluated to TRUE (i.e.     */
      /* there were no conflicts between variable bindings, */
      /* all tests were satisfied, etc.), then perform the  */
      /* appropriate action given the logic of this join.   */
      /*====================================================*/

      if (exprResult != FALSE)
        {
         /*==============================================*/
         /* Use the PPDrive routine when the join isn't  */
         /* associated with a not CE and it doesn't have */
         /* a join from the right.                       */
         /*==============================================*/

         PPDrive(theEnv,lhsBinds,rhsBinds,join);
        }

      /*====================================*/
      /* Move on to the next partial match. */
      /*====================================*/

      lhsBinds = lhsBinds->nextInNode;
     }

   return;
  }

/************************************************************/
/* NetworkAssertNegatedRight: Primary routine for filtering */
/*   a partial match through the join network from the RHS  */
/*   of a join connected to a not CE.                       */
/************************************************************/
static void NetworkAssertNegatedRight(
  void *theEnv,
  struct partialMatch *rhsBinds,
  struct joinNode *join)
  {
   struct partialMatch *lhsBinds = NULL, *nextBind;
   int exprResult;
   /*
   int frameSize = 0;
   struct compareFrame *staticFrame, *dynamicFrame;
   int needToEval;
   struct expr *leftHash, *rightHash;
*/
   /*=========================================================*/
   /* If an incremental reset is being performed and the join */
   /* is not part of the network to be reset, then return.    */
   /*=========================================================*/

#if (! BLOAD_ONLY) && (! RUN_TIME)
   if (EngineData(theEnv)->IncrementalResetInProgress && (join->initialize == FALSE)) return;
#endif

   /*==================================================*/
   /* Initialize some variables used to indicate which */
   /* side is being compared to the new partial match. */
   /*==================================================*/

   /*=====================================================*/
   /* The partial matches entering from the LHS of a join */
   /* are stored in the beta memory of the previous join  */
   /* (unless the current join is a join from the right   */
   /* or is attached to a not CE).                        */
   /*=====================================================*/

   lhsBinds = join->beta;
         
#if DEVELOPER
   if (lhsBinds != NULL)
     { EngineData(theEnv)->rightToLeftLoops++; }
#endif
    
   /*===================================================*/
   /* Compare each set of binds on the opposite side of */
   /* the join with the set of binds that entered this  */
   /* join. If the binds don't mismatch, then perform   */
   /* the appropriate action for the logic of the join. */
   /*===================================================*/

   while (lhsBinds != NULL)
     {
      nextBind = lhsBinds->nextInNode;
      
      /*===========================================================*/
      /* Initialize some variables pointing to the partial matches */
      /* in the LHS and RHS of the join.                           */
      /*===========================================================*/

      if ((lhsBinds->hashExpr == join->leftHash) &&
          (lhsBinds->hashValue != rhsBinds->hashValue))
        {
         lhsBinds = nextBind;
#if DEVELOPER
         EngineData(theEnv)->betaHashSkips++;
#endif
         continue;
        }

      /*===================================================*/
      /* If the join has no expression associated with it, */
      /* then the new partial match derived from the LHS   */
      /* and RHS partial matches is valid.                 */
      /*===================================================*/

      if (join->networkTest == NULL)
        { exprResult = TRUE; }

      /*=========================================================*/
      /* If the join has an expression associated with it, then  */
      /* evaluate the expression to determine if the new partial */
      /* match derived from the LHS and RHS partial matches is   */
      /* valid (i.e. variable bindings are consistent and        */
      /* predicate expressions evaluate to TRUE).                */
      /*=========================================================*/

      else
        {
#if DEVELOPER
         EngineData(theEnv)->rightToLeftComparisons++;
#endif
         exprResult = EvaluateJoinExpression(theEnv,join->networkTest,lhsBinds,rhsBinds,join);
         if (EvaluationData(theEnv)->EvaluationError)
           {
            if (join->patternIsNegated) exprResult = TRUE;
            SetEvaluationError(theEnv,FALSE);
           }

#if DEVELOPER
         if (exprResult)
           { EngineData(theEnv)->rightToLeftSucceeds++; }
#endif
        }

      /*====================================================*/
      /* If the join expression evaluated to TRUE (i.e.     */
      /* there were no conflicts between variable bindings, */
      /* all tests were satisfied, etc.), then perform the  */
      /* appropriate action given the logic of this join.   */
      /*====================================================*/

      if (exprResult != FALSE)
        {
         /*=====================================================*/
         /* Use the PNRDrive routine when the new partial match */
         /* enters from the RHS of the join and the join either */
         /* is associated with a not CE or has a join from the  */
         /* right.                                              */
         /*=====================================================*/

         PNRDrive(theEnv,join,lhsBinds,rhsBinds);
        }

      /*====================================*/
      /* Move on to the next partial match. */
      /*====================================*/

      lhsBinds = nextBind;
     }

   return;
  }

/**********************************************************/
/* NetworkAssertRightJFTR: Primary routine for filtering  */
/*   a partial match through the join network when a join */
/*   is entered from the right from another join.         */
/**********************************************************/
globle void NetworkAssertRightJFTR(
  void *theEnv,
  struct partialMatch *rhsBinds,
  struct joinNode *join)
  {
   struct partialMatch *lhsBinds = NULL, *nextBind;
   int exprResult;
   int i;

   /*=========================================================*/
   /* If an incremental reset is being performed and the join */
   /* is not part of the network to be reset, then return.    */
   /*=========================================================*/

#if (! BLOAD_ONLY) && (! RUN_TIME)
   if (EngineData(theEnv)->IncrementalResetInProgress && (join->initialize == FALSE)) return;
#endif

   /*=====================================================*/
   /* We need to look at the partial matches in the beta  */
   /* memory of this join to determine if the partial     */
   /* match entering from the right invalidates any of    */
   /* them.                                               */
   /*=====================================================*/

   lhsBinds = join->beta;
         
#if DEVELOPER
   if (lhsBinds != NULL)
     { EngineData(theEnv)->rightToLeftLoops++; }
#endif
    
   /*===================================================*/
   /* Compare each set of binds on the opposite side of */
   /* the join with the set of binds that entered this  */
   /* join. If the binds don't mismatch, then perform   */
   /* the appropriate action for the logic of the join. */
   /*===================================================*/

   while (lhsBinds != NULL)
     {
      nextBind = lhsBinds->nextInNode;
      
      /*=========================================================*/
      /* In the event that the join is a join from the right, it */
      /* must be checked that the RHS partial match is the same  */
      /* partial match that the LHS partial match was generated  */
      /* from. Each LHS partial match in a join from the right   */
      /* corresponds uniquely to a partial match from the RHS of */
      /* the join. To determine whether the LHS partial match is */
      /* the one associated with the RHS partial match, we       */
      /* compare the the entity addresses found in the partial   */
      /* matches to make sure they're equal.                     */
      /*=========================================================*/

      exprResult = TRUE;

      for (i = 0; i < (int) (lhsBinds->bcount - 1); i++)
        {
         if (lhsBinds->binds[i].gm.theMatch != rhsBinds->binds[i].gm.theMatch)
           {
            exprResult = FALSE;
            break;
           }
        }

      /*====================================================*/
      /* If the partial match from the right corresponds to */
      /* the partial match from the beta memory, then the   */
      /* partial match needs to be removed from the beta    */
      /* memory as it is no longer valid.                   */
      /*====================================================*/

      if (exprResult != FALSE)
        { PNRDrive(theEnv,join,lhsBinds,rhsBinds); }

      lhsBinds = nextBind;
     }

   return;
  }

/****************************************************/
/* NetworkAssertLeft: Primary routine for filtering */
/*   a partial match through the join network when  */
/*   entering through the left side of a join.      */
/****************************************************/
globle void NetworkAssertLeft(
  void *theEnv,
  struct partialMatch *lhsBinds,
  struct joinNode *join)
  {
   struct partialMatch *rhsBinds = NULL;
   struct partialMatch *newBinds;
   int exprResult;
   unsigned long hashValue;
   /*
   int frameSize = 0;
   struct compareFrame *staticFrame, *dynamicFrame;
   int needToEval;
   struct expr *leftHash, *rightHash;
*/
   /*=========================================================*/
   /* If an incremental reset is being performed and the join */
   /* is not part of the network to be reset, then return.    */
   /*=========================================================*/

#if (! BLOAD_ONLY) && (! RUN_TIME)
   if (EngineData(theEnv)->IncrementalResetInProgress && (join->initialize == FALSE)) return;
#endif

   /*=========================================================*/
   /* If the associated LHS pattern is a not CE or the join   */
   /* is a nand join, then we need an additional field in the */
   /* partial match to keep track of the pseudo fact if one   */
   /* is created. The partial match is automatically stored   */
   /* in the beta memory and the counterf slot is used to     */
   /* determine if it is an actual partial match. If counterf */
   /* is TRUE, there are one or more fact or instances        */
   /* keeping the not or nand join from being satisfied.      */
   /*=========================================================*/

   if ((join->patternIsNegated) || (join->joinFromTheRight))
     {
      newBinds = AddNotMatch(theEnv,lhsBinds,
                                (join->ruleToActivate == NULL) ? 0 : 1,
                                (int) join->logicalJoin);
      UpdatePMLinks(newBinds,lhsBinds,NULL,join,TRUE); /* Assume conflict and place in notBeta memory. */
      lhsBinds = newBinds;
     }

   /*==================================================*/
   /* Use a special routine if this is the first join. */
   /*==================================================*/

   if (join->firstJoin)
     {
      EmptyDrive(theEnv,join,lhsBinds);
      return;
     }

   /*==================================================*/
   /* Initialize some variables used to indicate which */
   /* side is being compared to the new partial match. */
   /*==================================================*/

   if (join->joinFromTheRight)
     { rhsBinds = ((struct joinNode *) join->rightSideEntryStructure)->beta; }
   else
     { 
      if (join->leftHash != NULL)
        { 
         hashValue = BetaMemoryHashValue(theEnv,join->leftHash,lhsBinds,NULL,join,NULL);
         lhsBinds->hashValue = hashValue;  
         lhsBinds->hashExpr = join->leftHash;
        }
      else
        { hashValue = 0; }
           
      if (DefruleData(theEnv)->JoinHashingFlag)
        { rhsBinds = GetAlphaMemory(theEnv,(struct patternNodeHeader *) join->rightSideEntryStructure,hashValue); }
      else
        { rhsBinds = GetAlphaMemory(theEnv,(struct patternNodeHeader *) join->rightSideEntryStructure,0); }
     }
       
#if DEVELOPER
   if (rhsBinds != NULL)
     { EngineData(theEnv)->leftToRightLoops++; }
#endif
    
   /*===================================================*/
   /* Compare each set of binds on the opposite side of */
   /* the join with the set of binds that entered this  */
   /* join. If the binds don't mismatch, then perform   */
   /* the appropriate action for the logic of the join. */
   /*===================================================*/

   while (rhsBinds != NULL)
     {
      /*========================================================*/
      /* If the join has no expression associated with it, then */
      /* the new partial match derived from the LHS and RHS     */
      /* partial matches is valid. In the event that the join   */
      /* is a join from the right, it must also be checked that */
      /* the RHS partial match is the same partial match that   */
      /* the LHS partial match was generated from. Each LHS     */
      /* partial match in a join from the right corresponds     */
      /* uniquely to a partial match from the RHS of the join.  */
      /* To determine whether the LHS partial match is the one  */
      /* associated with the RHS partial match, we compare the  */
      /* the entity addresses found in the partial matches to   */
      /* make sure they're equal.                               */
      /*========================================================*/

      if (join->networkTest == NULL)
        {
         exprResult = TRUE;
         if (join->joinFromTheRight)
           {
            int i;

            for (i = 0; i < (int) (lhsBinds->bcount - 1); i++)
              {
               if (lhsBinds->binds[i].gm.theMatch != rhsBinds->binds[i].gm.theMatch)
                 {
                  exprResult = FALSE;
                  break;
                 }
              }
           }
        }

      /*=========================================================*/
      /* If the join has an expression associated with it, then  */
      /* evaluate the expression to determine if the new partial */
      /* match derived from the LHS and RHS partial matches is   */
      /* valid (i.e. variable bindings are consistent and        */
      /* predicate expressions evaluate to TRUE).                */
      /*=========================================================*/

      else
        {
#if DEVELOPER
         EngineData(theEnv)->leftToRightComparisons++;
#endif
         exprResult = EvaluateJoinExpression(theEnv,join->networkTest,lhsBinds,rhsBinds,join);
         if (EvaluationData(theEnv)->EvaluationError)
           {
            if (join->patternIsNegated) exprResult = TRUE;
            SetEvaluationError(theEnv,FALSE);
           }

#if DEVELOPER
         if (exprResult)
           { EngineData(theEnv)->leftToRightSucceeds++; }
#endif
        }

      /*====================================================*/
      /* If the join expression evaluated to TRUE (i.e.     */
      /* there were no conflicts between variable bindings, */
      /* all tests were satisfied, etc.), then perform the  */
      /* appropriate action given the logic of this join.   */
      /*====================================================*/

      if (exprResult != FALSE)
        {
         /*==============================================*/
         /* Use the PPDrive routine when the join isn't  */
         /* associated with a not CE and it doesn't have */
         /* a join from the right.                       */
         /*==============================================*/

         if ((join->patternIsNegated == FALSE) &&
             (join->joinFromTheRight == FALSE))
           { PPDrive(theEnv,lhsBinds,rhsBinds,join); }

         /*===========================================================*/
         /* If the new partial match entered from the LHS of the join */
         /* and the join is either associated with a not CE or the    */
         /* join has a join from the right, then mark the LHS partial */
         /* match indicating that there is a RHS partial match        */
         /* preventing this join from being satisfied. Once this has  */
         /* happened, the other RHS partial matches don't have to be  */
         /* tested since it only takes one partial match to prevent   */
         /* the LHS from being satisfied.                             */
         /*===========================================================*/

         else
           {
            lhsBinds->binds[lhsBinds->bcount - 1].gm.theValue = (void *) rhsBinds;
            UpdatePMAlphaLinks(lhsBinds,rhsBinds);
            rhsBinds = NULL;
            continue; /* TBD should be break */
           }
        }

      /*====================================*/
      /* Move on to the next partial match. */
      /*====================================*/

      rhsBinds = rhsBinds->nextInNode;
     }

   /*==========================================================*/
   /* Remove the static comparison frame if one was generated. */
   /*==========================================================*/
   /*
   if (staticFrame != NULL)
     { genfree(theEnv,staticFrame,frameSize * sizeof(struct compareFrame)); }
*/

   /*==================================================================*/
   /* If a join with an associated not CE or join from the right was   */
   /* entered from the LHS side of the join, and the join expression   */
   /* failed for all sets of matches for the new bindings on the LHS   */
   /* side (there was no RHS partial match preventing the LHS partial  */
   /* match from being satisfied), then the LHS partial match appended */
   /* with an pseudo-fact that represents the instance of the not      */
   /* pattern or join from the right that was satisfied should be sent */
   /* to the joins below this join.                                    */
   /*==================================================================*/

   if ((join->patternIsNegated || join->joinFromTheRight) &&
       (lhsBinds->binds[lhsBinds->bcount - 1].gm.theValue == NULL))
     { PNLDrive(theEnv,join,lhsBinds); }

   return;
  }

/*******************************************************/
/* EvaluateJoinExpression: Evaluates join expressions. */
/*   Performs a faster evaluation for join expressions */
/*   than if EvaluateExpression was used directly.     */
/*******************************************************/
globle intBool EvaluateJoinExpression(
  void *theEnv,
  struct expr *joinExpr,
  struct partialMatch *lbinds,
  struct partialMatch *rbinds,
  struct joinNode *joinPtr)
  {
   DATA_OBJECT theResult;
   int andLogic, result = TRUE;
   struct partialMatch *oldLHSBinds;
   struct partialMatch *oldRHSBinds;
   struct joinNode *oldJoin;

   /*======================================*/
   /* A NULL expression evaluates to TRUE. */
   /*======================================*/

   if (joinExpr == NULL) return(TRUE);

   /*=========================================*/
   /* Initialize some of the global variables */
   /* used when evaluating expressions.       */
   /*=========================================*/

   oldLHSBinds = EngineData(theEnv)->GlobalLHSBinds;
   oldRHSBinds = EngineData(theEnv)->GlobalRHSBinds;
   oldJoin = EngineData(theEnv)->GlobalJoin;
   EngineData(theEnv)->GlobalLHSBinds = lbinds;
   EngineData(theEnv)->GlobalRHSBinds = rbinds;
   EngineData(theEnv)->GlobalJoin = joinPtr;

   /*=====================================================*/
   /* Partial matches stored in joins that are associated */
   /* with a not CE contain an additional slot shouldn't  */
   /* be considered when evaluating expressions. Since    */
   /* joins that have joins from the right don't have any */
   /* expression, we don't have to do this for partial    */
   /* matches contained in these joins.                   */
   /*=====================================================*/

   if (joinPtr->patternIsNegated) lbinds->bcount--;

   /*====================================================*/
   /* Initialize some variables which allow this routine */
   /* to avoid calling the "and" and "or" functions if   */
   /* they are the first part of the expression to be    */
   /* evaluated. Most of the join expressions do not use */
   /* deeply nested and/or functions so this technique   */
   /* speeds up evaluation.                              */
   /*====================================================*/

   if (joinExpr->value == ExpressionData(theEnv)->PTR_AND)
     {
      andLogic = TRUE;
      joinExpr = joinExpr->argList;
     }
   else if (joinExpr->value == ExpressionData(theEnv)->PTR_OR)
     {
      andLogic = FALSE;
      joinExpr = joinExpr->argList;
     }
   else
     { andLogic = TRUE; }

   /*=========================================*/
   /* Evaluate each of the expressions linked */
   /* together in the join expression.        */
   /*=========================================*/

   while (joinExpr != NULL)
     {
      /*================================*/
      /* Evaluate a primitive function. */
      /*================================*/

      if ((EvaluationData(theEnv)->PrimitivesArray[joinExpr->type] == NULL) ?
          FALSE :
          EvaluationData(theEnv)->PrimitivesArray[joinExpr->type]->evaluateFunction != NULL)
        {
         struct expr *oldArgument;

         oldArgument = EvaluationData(theEnv)->CurrentExpression;
         EvaluationData(theEnv)->CurrentExpression = joinExpr;
         result = (*EvaluationData(theEnv)->PrimitivesArray[joinExpr->type]->evaluateFunction)(theEnv,joinExpr->value,&theResult);
         EvaluationData(theEnv)->CurrentExpression = oldArgument;
        }

      /*=============================*/
      /* Evaluate the "or" function. */
      /*=============================*/

      else if (joinExpr->value == ExpressionData(theEnv)->PTR_OR)
        {
         result = FALSE;
         if (EvaluateJoinExpression(theEnv,joinExpr,lbinds,rbinds,joinPtr) == TRUE)
           {
            if (EvaluationData(theEnv)->EvaluationError)
              {
               if (joinPtr->patternIsNegated) lbinds->bcount++;
               EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
               EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
               EngineData(theEnv)->GlobalJoin = oldJoin;
               return(FALSE);
              }
            result = TRUE;
           }
         else if (EvaluationData(theEnv)->EvaluationError)
           {
            if (joinPtr->patternIsNegated) lbinds->bcount++;
            EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
            EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
            EngineData(theEnv)->GlobalJoin = oldJoin;
            return(FALSE);
           }
        }

      /*==============================*/
      /* Evaluate the "and" function. */
      /*==============================*/

      else if (joinExpr->value == ExpressionData(theEnv)->PTR_AND)
        {
         result = TRUE;
         if (EvaluateJoinExpression(theEnv,joinExpr,lbinds,rbinds,joinPtr) == FALSE)
           {
            if (EvaluationData(theEnv)->EvaluationError)
              {
               if (joinPtr->patternIsNegated) lbinds->bcount++;
               EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
               EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
               EngineData(theEnv)->GlobalJoin = oldJoin;
               return(FALSE);
              }
            result = FALSE;
           }
         else if (EvaluationData(theEnv)->EvaluationError)
           {
            if (joinPtr->patternIsNegated) lbinds->bcount++;
            EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
            EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
            EngineData(theEnv)->GlobalJoin = oldJoin;
            return(FALSE);
           }
        }

      /*==========================================================*/
      /* Evaluate all other expressions using EvaluateExpression. */
      /*==========================================================*/

      else
        {
         EvaluateExpression(theEnv,joinExpr,&theResult);

         if (EvaluationData(theEnv)->EvaluationError)
           {
            JoinNetErrorMessage(theEnv,joinPtr);
            if (joinPtr->patternIsNegated) lbinds->bcount++;
            EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
            EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
            EngineData(theEnv)->GlobalJoin = oldJoin;
            return(FALSE);
           }

         if ((theResult.value == EnvFalseSymbol(theEnv)) && (theResult.type == SYMBOL))
           { result = FALSE; }
         else
           { result = TRUE; }
        }

      /*====================================*/
      /* Handle the short cut evaluation of */
      /* the "and" and "or" functions.      */
      /*====================================*/

      if ((andLogic == TRUE) && (result == FALSE))
        {
         if (joinPtr->patternIsNegated) lbinds->bcount++;
         EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
         EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
         EngineData(theEnv)->GlobalJoin = oldJoin;
         return(FALSE);
        }
      else if ((andLogic == FALSE) && (result == TRUE))
        {
         if (joinPtr->patternIsNegated) lbinds->bcount++;
         EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
         EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
         EngineData(theEnv)->GlobalJoin = oldJoin;
         return(TRUE);
        }

      /*==============================================*/
      /* Move to the next expression to be evaluated. */
      /*==============================================*/

      joinExpr = joinExpr->nextArg;
     }

   /*=======================================*/
   /* Restore some of the global variables. */
   /*=======================================*/

   EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
   EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
   EngineData(theEnv)->GlobalJoin = oldJoin;

   /*=====================================*/
   /* Restore the count value for the LHS */
   /* binds if it had to be modified.     */
   /*=====================================*/

   if (joinPtr->patternIsNegated) lbinds->bcount++;

   /*=================================================*/
   /* Return the result of evaluating the expression. */
   /*=================================================*/

   return(result);
  }

/*******************************************************/
/* BetaMemoryHashValue:     */
/*******************************************************/
static unsigned long BetaMemoryHashValue(
  void *theEnv,
  struct expr *hashExpr,
  struct partialMatch *lbinds,
  struct partialMatch *rbinds,
  struct joinNode *joinPtr,
  struct compareFrame *frame)
  {
   DATA_OBJECT theResult;
   struct partialMatch *oldLHSBinds;
   struct partialMatch *oldRHSBinds;
   struct joinNode *oldJoin;
   int i = 0;
   unsigned long hashValue = 0;
   unsigned long multiplier = 1;
   
   /*======================================*/
   /* A NULL expression evaluates to zero. */
   /*======================================*/

   if (hashExpr == NULL) return(0);

   /*=========================================*/
   /* Initialize some of the global variables */
   /* used when evaluating expressions.       */
   /*=========================================*/

   oldLHSBinds = EngineData(theEnv)->GlobalLHSBinds;
   oldRHSBinds = EngineData(theEnv)->GlobalRHSBinds;
   oldJoin = EngineData(theEnv)->GlobalJoin;
   EngineData(theEnv)->GlobalLHSBinds = lbinds;
   EngineData(theEnv)->GlobalRHSBinds = rbinds;
   EngineData(theEnv)->GlobalJoin = joinPtr;

   /*=========================================*/
   /* Evaluate each of the expressions linked */
   /* together in the join expression.        */
   /*=========================================*/

   while (hashExpr != NULL)
     {
      /*================================*/
      /* Evaluate a primitive function. */
      /*================================*/

      if ((EvaluationData(theEnv)->PrimitivesArray[hashExpr->type] == NULL) ?
          FALSE :
          EvaluationData(theEnv)->PrimitivesArray[hashExpr->type]->evaluateFunction != NULL)
        {
         struct expr *oldArgument;

         oldArgument = EvaluationData(theEnv)->CurrentExpression;
         EvaluationData(theEnv)->CurrentExpression = hashExpr;
         (*EvaluationData(theEnv)->PrimitivesArray[hashExpr->type]->evaluateFunction)(theEnv,hashExpr->value,&theResult);
         EvaluationData(theEnv)->CurrentExpression = oldArgument;
        }

      /*==========================================================*/
      /* Evaluate all other expressions using EvaluateExpression. */
      /*==========================================================*/

      else
        { EvaluateExpression(theEnv,hashExpr,&theResult); }

      switch (theResult.type)
        {
         case STRING:
         case SYMBOL:
         case INSTANCE_NAME:
           hashValue += (((SYMBOL_HN *) theResult.value)->bucket * multiplier); /* TBD Group with alpha computation */
           break;
             
         case INTEGER:
            hashValue += (((INTEGER_HN *) theResult.value)->bucket * multiplier);
            break;
             
         case FLOAT:
           hashValue += (((FLOAT_HN *) theResult.value)->bucket * multiplier);
           break;
        }

      /*==============================================*/
      /* Move to the next expression to be evaluated. */
      /*==============================================*/

      hashExpr = hashExpr->nextArg;
      multiplier = multiplier * 509;
      i++;
     }

   /*=======================================*/
   /* Restore some of the global variables. */
   /*=======================================*/

   EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
   EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
   EngineData(theEnv)->GlobalJoin = oldJoin;

   /*=================================================*/
   /* Return the result of evaluating the expression. */
   /*=================================================*/

   return(hashValue);
  }

/*******************************************************************/
/* PPDrive: Handles the merging of an alpha memory partial match   */
/*   with a beta memory partial match for a join that has positive */
/*   LHS entry and positive RHS entry. The partial matches being   */
/*   merged have previously been checked to determine that they    */
/*   satisify the constraints for the join. Once merged, the new   */
/*   partial match is sent to each child join of the join from     */
/*   which the merge took place.                                   */
/*******************************************************************/
static void PPDrive(
  void *theEnv,
  struct partialMatch *lhsBinds,
  struct partialMatch *rhsBinds,
  struct joinNode *join)
  {
   struct partialMatch *linker;
   struct joinNode *listOfJoins;
   
   /*==================================================*/
   /* Merge the alpha and beta memory partial matches. */
   /*==================================================*/

   linker = MergePartialMatches(theEnv,lhsBinds,rhsBinds,
                                (join->ruleToActivate == NULL) ? 0 : 1,
                                (int) join->logicalJoin);

   /*=======================================================*/
   /* Add the partial match to the beta memory of the join. */
   /*=======================================================*/

   UpdatePMLinks(linker,lhsBinds,rhsBinds,join,FALSE);

   /*====================================================*/
   /* Activate the rule satisfied by this partial match. */
   /*====================================================*/

   if (join->ruleToActivate != NULL) AddActivation(theEnv,join->ruleToActivate,linker);

   /*================================================*/
   /* Send the new partial match to all child joins. */
   /*================================================*/

   listOfJoins = join->nextLevel;
   if (listOfJoins != NULL)
     {
      if (((struct joinNode *) (listOfJoins->rightSideEntryStructure)) == join)
        { NetworkAssertRightJFTR(theEnv,linker,listOfJoins); }
      else while (listOfJoins != NULL)
        {
         NetworkAssertLeft(theEnv,linker,listOfJoins);
         listOfJoins = listOfJoins->rightDriveNode;
        }
     }

   return;
  }

/**********************************************************************/
/* PNRDrive: Handles the entry of a partial match from the RHS of a   */
/*   join that has positive LHS entry and negative RHS entry (meaning */
/*   the conditional element associated with this join is a not       */
/*   conditional element). Entry of the alpha memory partial match    */
/*   will cause the counterf value of the associated beta memory      */
/*   partial match to be set. This in turn may cause partial matches  */
/*   associated with the beta memory partial match to be removed from */
/*   the network.                                                     */
/**********************************************************************/
static void PNRDrive(
  void *theEnv,
  struct joinNode *join,
  struct partialMatch *lhsBinds,
  struct partialMatch *rhsBinds)
  {
   /*==================================================*/
   /* If the partial match already has a partial match */
   /* in the alpha memory which prevents it from being */
   /* satisfied, then don't do anything.               */
   /*==================================================*/

   if (lhsBinds->counterf == TRUE) return;

   /*=================================================*/
   /* Set the counterf flag to indicate that an alpha */
   /* memory partial match is preventing the beta     */
   /* memory partial match from being satisfied.      */
   /*=================================================*/

   lhsBinds->counterf = TRUE;

   /*===================================================================*/
   /* If the partial match caused an activation, remove the activation. */
   /*===================================================================*/

   if ((lhsBinds->activationf) ?
       (lhsBinds->binds[lhsBinds->bcount].gm.theValue != NULL) : FALSE)
     { RemoveActivation(theEnv,(struct activation *) lhsBinds->binds[lhsBinds->bcount].gm.theValue,TRUE,TRUE); }

   /*===========================================================*/
   /* The counterf flag was FALSE. This means that a pointer to */
   /* the pseudo-fact matching the not CE is stored directly in */
   /* the partial match. Determine the ID of this pseudo-fact   */
   /* and remove all partial matches from descendent joins that */
   /* contain the ID.                                           */
   /*===========================================================*/

   if (join->joinFromTheRight) /* GDR 111599 #834 Begin */
     {/* TBD Is this necessary?
      RetractCheckDriveRetractions(theEnv,lhsBinds->binds[lhsBinds->bcount - 1].gm.theMatch,
                                   (int) join->depth-1); */
     }                         /* GDR 111599 #834 End */

   if (lhsBinds->children != NULL)
     { PosEntryRetractBeta(theEnv,lhsBinds,lhsBinds->children,NULL); }
     
   /*=========================================================================*/
   /* Remove any logical dependency links associated with this partial match. */
   /*=========================================================================*/

   if (lhsBinds->dependentsf) RemoveLogicalSupport(theEnv,lhsBinds);

   /*========================================================*/
   /* Store the partial match from the alpha memory that is  */
   /* preventing the LHS partial match from being satisfied. */
   /*========================================================*/

   lhsBinds->binds[lhsBinds->bcount - 1].gm.theValue = (void *) rhsBinds;

   UnlinkBetaPMFromNode(join,lhsBinds,FALSE); /* Remove from Beta. */
   LinkBetaPMToNode(join,lhsBinds,TRUE);      /* Add to notBeta. */
   UpdatePMAlphaLinks(lhsBinds,rhsBinds);
  }

/********************************************************************/
/* PNLDrive: Handles the entry of a partial match from the LHS of a */
/*   join that has positive LHS entry and negative RHS entry        */
/*   (meaning the conditional element associated with this join is  */
/*   a not conditional element). An new partial match is created by */
/*   combining the match from the beta memory with a "pseudo"       */
/*   partial match corresponding to the facts which didn't match    */
/*   the not CE. Once merged, the new partial match is sent to each */
/*   child join of the join from which the merge took place.        */
/********************************************************************/
globle void PNLDrive(
  void *theEnv,
  struct joinNode *join,
  struct partialMatch *binds)
  {
   struct joinNode *listOfJoins;

    /*===============================================*/
    /* Store the pointer to the pseudo-fact directly */
    /* in the beta memory partial match.             */
    /*===============================================*/

    binds->counterf = FALSE;
    binds->binds[binds->bcount - 1].gm.theMatch = NULL;

    UnlinkBetaPMFromNode(join,binds,TRUE); /* Remove from notBeta. */
    LinkBetaPMToNode(join,binds,FALSE);    /* Add to beta. */
            
   /*====================================================*/
   /* Activate the rule satisfied by this partial match. */
   /*====================================================*/

   if (join->ruleToActivate != NULL) AddActivation(theEnv,join->ruleToActivate,binds);

    /*========================================================*/
    /* Send the merged partial match to all descendent joins. */
    /*========================================================*/

    listOfJoins = join->nextLevel;
    if (listOfJoins != NULL)
      {
       if (((struct joinNode *) (listOfJoins->rightSideEntryStructure)) == join)
         { NetworkAssertRightJFTR(theEnv,binds,listOfJoins); }
       else while (listOfJoins != NULL)
         {
          NetworkAssertLeft(theEnv,binds,listOfJoins);
          listOfJoins = listOfJoins->rightDriveNode;
         }
      }
   }

/***************************************************************/
/* EmptyDrive: Handles the entry of a alpha memory partial     */
/*   match from the RHS of a join that is the first join of    */
/*   a rule (i.e. a join that cannot be entered from the LHS). */
/***************************************************************/
static void EmptyDrive(
  void *theEnv,
  struct joinNode *join,
  struct partialMatch *rhsBinds)
  {
   struct partialMatch *linker;
   struct joinNode *listOfJoins;
   int joinExpr;

   /*======================================================*/
   /* Determine if the alpha memory partial match satifies */
   /* the join expression. If it doesn't then no further   */
   /* action is taken.                                     */
   /*======================================================*/

   if (join->networkTest != NULL)
     {

#if DEVELOPER
      if (join->networkTest)
        { EngineData(theEnv)->rightToLeftComparisons++; }
#endif

      joinExpr = EvaluateJoinExpression(theEnv,join->networkTest,NULL,rhsBinds,join);
      EvaluationData(theEnv)->EvaluationError = FALSE;
      if (joinExpr == FALSE) return;
     }

   /*===========================================================*/
   /* The first join of a rule cannot be connected to a NOT CE. */
   /*===========================================================*/

   if (join->patternIsNegated == TRUE)
     {
      SystemError(theEnv,"DRIVE",2);
      EnvExitRouter(theEnv,EXIT_FAILURE);
     }

   /*=========================================================*/
   /* If the join's RHS entry is associated with a pattern CE */
   /* (positive entry), then copy the alpha memory partial    */
   /* match and send it to all child joins.                   */
   /*=========================================================*/

   linker = CopyPartialMatch(theEnv,rhsBinds,
                             (join->ruleToActivate == NULL) ? 0 : 1,
                             (int) join->logicalJoin);

   /*=======================================================*/
   /* Add the partial match to the beta memory of the join. */
   /*=======================================================*/

   UpdatePMLinks(linker,NULL,rhsBinds,join,FALSE);

   /*====================================================*/
   /* Activate the rule satisfied by this partial match. */
   /*====================================================*/

   if (join->ruleToActivate != NULL) AddActivation(theEnv,join->ruleToActivate,linker);

   /*============================================*/
   /* Send the partial match to all child joins. */
   /*============================================*/

   listOfJoins = join->nextLevel;
   while (listOfJoins != NULL)
     {
      NetworkAssertLeft(theEnv,linker,listOfJoins);
      listOfJoins = listOfJoins->rightDriveNode;
     }
  }

/***************************************************************/
/* RetrieveStaticBinds:  */
/***************************************************************/
static struct compareFrame *RetrieveStaticBinds(
  void *theEnv,
  struct partialMatch *theBinds,
  int enterDirection,
  struct joinNode *joinPtr,
  unsigned long *hashValue,
  int *frameSize)
  {
   int count = 0;
   struct expr *exprPtr;
   struct compareFrame *frame = NULL;
   
   if (enterDirection == LHS)
     {
      for (exprPtr = joinPtr->leftHash; exprPtr != NULL; exprPtr = exprPtr->nextArg)
        { count++; }
      
      *frameSize = count;
      frame = genalloc(theEnv,count * sizeof(struct compareFrame));
        
      *hashValue = BetaMemoryHashValue(theEnv,joinPtr->leftHash,theBinds,NULL,joinPtr,frame);
     }
   else
     {
      for (exprPtr = ((struct patternNodeHeader *) joinPtr->rightSideEntryStructure)->rightHash; 
           exprPtr != NULL; 
           exprPtr = exprPtr->nextArg)
        { count++; }

      *frameSize = count;
      frame = genalloc(theEnv,count * sizeof(struct compareFrame));

      *hashValue = BetaMemoryHashValue(theEnv,((struct patternNodeHeader *) joinPtr->rightSideEntryStructure)->rightHash,NULL,theBinds,joinPtr,frame);
     }
     
   return frame;
  }

/***************************************************************/
/* CreateDynamicBinds:  */
/***************************************************************/
static struct compareFrame *CreateDynamicBinds(
  void *theEnv,
  int frameSize)
  {
   struct compareFrame *frame = NULL;
   int i;
   
   if (frameSize == 0)
     { return NULL; }
     
   frame = genalloc(theEnv,frameSize * sizeof(struct compareFrame));
        
   for (i = 0; i < frameSize; i++)
     {
      frame[i].slotValue.type = RVOID;
     }
     
   return frame;
  }

/***************************************************************/
/* CompareFrames: */
/***************************************************************/
static int CompareFrames(
  void *theEnv,
  struct joinNode *join,
  struct compareFrame *staticFrame,
  struct compareFrame *dynamicFrame,
  struct partialMatch *dynamicMatch,
  struct partialMatch *lhsBinds,
  struct partialMatch *rhsBinds,
  struct expr *dynamicHash,
  int frameSize)
  {
   struct partialMatch *oldLHSBinds;
   struct partialMatch *oldRHSBinds;
   struct joinNode *oldJoin;
   int i;
   struct expr *oldArgument;
   
   /*=========================================*/
   /* Initialize some of the global variables */
   /* used when evaluating expressions.       */
   /*=========================================*/

   oldLHSBinds = EngineData(theEnv)->GlobalLHSBinds;
   oldRHSBinds = EngineData(theEnv)->GlobalRHSBinds;
   oldJoin = EngineData(theEnv)->GlobalJoin;
   EngineData(theEnv)->GlobalLHSBinds = lhsBinds;
   EngineData(theEnv)->GlobalRHSBinds = rhsBinds;
   EngineData(theEnv)->GlobalJoin = join;

   for (i = 0; i < frameSize; i++)
     {
      oldArgument = EvaluationData(theEnv)->CurrentExpression;
      EvaluationData(theEnv)->CurrentExpression = dynamicHash;
      (*EvaluationData(theEnv)->PrimitivesArray[dynamicHash->type]->evaluateFunction)(theEnv,dynamicHash->value,&dynamicFrame[i].slotValue);
      EvaluationData(theEnv)->CurrentExpression = oldArgument;

      if ((staticFrame[i].slotValue.type != dynamicFrame[i].slotValue.type) ||
          (staticFrame[i].slotValue.value != dynamicFrame[i].slotValue.value))
        {
         EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
         EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
         EngineData(theEnv)->GlobalJoin = oldJoin;
         return(FALSE);
        }
        
      dynamicHash = dynamicHash->nextArg;      
     }

   /*=======================================*/
   /* Restore some of the global variables. */
   /*=======================================*/

   EngineData(theEnv)->GlobalLHSBinds = oldLHSBinds;
   EngineData(theEnv)->GlobalRHSBinds = oldRHSBinds;
   EngineData(theEnv)->GlobalJoin = oldJoin;
     
   return TRUE;
  }
  
/********************************************************************/
/* JoinNetErrorMessage: Prints an informational message indicating  */
/*   which join of a rule generated an error when a join expression */
/*   was being evaluated.                                           */
/********************************************************************/
static void JoinNetErrorMessage(
  void *theEnv,
  struct joinNode *joinPtr)
  {
   char buffer[60];

   PrintErrorID(theEnv,"DRIVE",1,TRUE);
   EnvPrintRouter(theEnv,WERROR,"This error occurred in the join network\n");

   sprintf(buffer,"   Problem resides in join #%d in rule(s):\n",joinPtr->depth);
   EnvPrintRouter(theEnv,WERROR,buffer);

   TraceErrorToRule(theEnv,joinPtr,"      ");
   EnvPrintRouter(theEnv,WERROR,"\n");
  }

#endif /* DEFRULE_CONSTRUCT */
